/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.layout;

import com.google.common.collect.FluentIterable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.vecmath.Point2d;
import javax.vecmath.Vector2d;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.geometry.GeometryUtil;
import org.openscience.cdk.graph.ConnectivityChecker;
import org.openscience.cdk.graph.Cycles;
import org.openscience.cdk.graph.GraphUtil;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IAtomContainerSet;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemObjectBuilder;
import org.openscience.cdk.interfaces.IPseudoAtom;
import org.openscience.cdk.interfaces.IRing;
import org.openscience.cdk.interfaces.IRingSet;
import org.openscience.cdk.isomorphism.AtomMatcher;
import org.openscience.cdk.isomorphism.BondMatcher;
import org.openscience.cdk.isomorphism.Pattern;
import org.openscience.cdk.isomorphism.VentoFoggia;
import org.openscience.cdk.layout.AtomPlacer;
import org.openscience.cdk.layout.CorrectGeometricConfiguration;
import org.openscience.cdk.layout.HydrogenPlacer;
import org.openscience.cdk.layout.IdentityTemplateLibrary;
import org.openscience.cdk.layout.LayoutRefiner;
import org.openscience.cdk.layout.MacroCycleLayout;
import org.openscience.cdk.layout.NonplanarBonds;
import org.openscience.cdk.layout.RingPlacer;
import org.openscience.cdk.layout.TemplateHandler;
import org.openscience.cdk.ringsearch.RingPartitioner;
import org.openscience.cdk.sgroup.Sgroup;
import org.openscience.cdk.sgroup.SgroupBracket;
import org.openscience.cdk.sgroup.SgroupKey;
import org.openscience.cdk.sgroup.SgroupType;
import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;
import org.openscience.cdk.tools.manipulator.AtomContainerManipulator;
import org.openscience.cdk.tools.manipulator.AtomContainerSetManipulator;
import org.openscience.cdk.tools.manipulator.RingSetManipulator;

public class StructureDiagramGenerator {
    private static ILoggingTool logger = LoggingToolFactory.createLoggingTool(StructureDiagramGenerator.class);
    public static final double DEFAULT_BOND_LENGTH = 1.5;
    private IAtomContainer molecule;
    private IRingSet sssr;
    private double bondLength = 1.5;
    private Vector2d firstBondVector;
    private RingPlacer ringPlacer = new RingPlacer();
    private AtomPlacer atomPlacer = new AtomPlacer();
    private MacroCycleLayout macroPlacer = null;
    private List<IRingSet> ringSystems = null;
    private boolean useIdentTemplates = true;
    private boolean selectOrientation = true;
    private IdentityTemplateLibrary identityLibrary;
    public static Vector2d DEFAULT_BOND_VECTOR = new Vector2d(0.0, 1.0);
    private static IdentityTemplateLibrary DEFAULT_TEMPLATE_LIBRARY = IdentityTemplateLibrary.loadFromResource("custom-templates.smi").add(IdentityTemplateLibrary.loadFromResource("chebi-ring-templates.smi"));
    private static final String FRAGMENT_CHARGE = "FragmentCharge";

    public StructureDiagramGenerator() {
        this(DEFAULT_TEMPLATE_LIBRARY);
    }

    private StructureDiagramGenerator(IdentityTemplateLibrary identityLibrary) {
        this.identityLibrary = identityLibrary;
    }

    public StructureDiagramGenerator(IAtomContainer molecule) {
        this();
        this.setMolecule(molecule, false);
    }

    public void setMolecule(IAtomContainer mol, boolean clone) {
        IAtom atom = null;
        if (clone) {
            try {
                this.molecule = mol.clone();
            }
            catch (CloneNotSupportedException e) {
                logger.error("Should clone, but exception occured: ", e.getMessage());
                logger.debug(e);
            }
        } else {
            this.molecule = mol;
        }
        for (int f = 0; f < this.molecule.getAtomCount(); ++f) {
            atom = this.molecule.getAtom(f);
            atom.setPoint2d(null);
            atom.setFlag(1, false);
            atom.setFlag(16, false);
            atom.setFlag(2, false);
            atom.setFlag(8, false);
        }
        this.atomPlacer.setMolecule(this.molecule);
        this.ringPlacer.setMolecule(this.molecule);
        this.ringPlacer.setAtomPlacer(this.atomPlacer);
        this.macroPlacer = new MacroCycleLayout(mol);
        this.selectOrientation = true;
    }

    @Deprecated
    public void setUseTemplates(boolean useTemplates) {
    }

    public void setUseIdentityTemplates(boolean use) {
        this.useIdentTemplates = use;
    }

    @Deprecated
    public boolean getUseTemplates() {
        return false;
    }

    @Deprecated
    public void setTemplateHandler(TemplateHandler templateHandler) {
        IdentityTemplateLibrary lib = templateHandler.toIdentityTemplateLibrary();
        lib.add(this.identityLibrary);
        this.identityLibrary = lib;
    }

    @Deprecated
    public TemplateHandler getTemplateHandler() {
        return null;
    }

    public void setMolecule(IAtomContainer molecule) {
        this.setMolecule(molecule, true);
    }

    public IAtomContainer getMolecule() {
        return this.molecule;
    }

    public void generateExperimentalCoordinates() throws CDKException {
        this.generateExperimentalCoordinates(DEFAULT_BOND_VECTOR);
    }

    public void generateExperimentalCoordinates(Vector2d firstBondVector) throws CDKException {
        IAtomContainer original = this.molecule;
        IAtomContainer shallowCopy = this.molecule.getBuilder().newInstance(IAtomContainer.class, this.molecule);
        for (IAtom curAtom : shallowCopy.atoms()) {
            if (!curAtom.getSymbol().equals("H") || shallowCopy.getConnectedBondsCount(curAtom) >= 2) continue;
            shallowCopy.removeAtomAndConnectedElectronContainers(curAtom);
            curAtom.setPoint2d(null);
        }
        this.molecule = shallowCopy;
        this.generateCoordinates(firstBondVector);
        double bondLength = GeometryUtil.getBondLengthAverage(this.molecule);
        HydrogenPlacer hPlacer = new HydrogenPlacer();
        this.molecule = original;
        hPlacer.placeHydrogens2D(this.molecule, bondLength);
    }

    public void generateCoordinates(Vector2d firstBondVector) throws CDKException {
        this.generateCoordinates(firstBondVector, false, false);
    }

    private void generateCoordinates(Vector2d firstBondVector, boolean isConnected, boolean isSubLayout) throws CDKException {
        IAtomContainerSet frags;
        int safetyCounter = 0;
        logger.debug("Entry point of generateCoordinates()");
        logger.debug("We have a molecules with " + this.molecule.getAtomCount() + " atoms.");
        if (this.molecule.getAtomCount() == 1) {
            this.molecule.getAtom(0).setPoint2d(new Point2d(0.0, 0.0));
            return;
        }
        if (this.molecule.getBondCount() == 1) {
            double xOffset = 0.0;
            for (IAtom atom : this.molecule.atoms()) {
                atom.setPoint2d(new Point2d(xOffset, 0.0));
                xOffset += this.bondLength;
            }
            return;
        }
        if (!isConnected && (frags = ConnectivityChecker.partitionIntoMolecules(this.molecule)).getAtomContainerCount() > 1) {
            this.generateFragmentCoordinates(this.molecule, this.toList(frags));
            this.molecule = this.molecule;
            return;
        }
        int nrOfEdges = this.molecule.getBondCount();
        this.firstBondVector = firstBondVector;
        int expectedRingCount = nrOfEdges - this.molecule.getAtomCount() + 1;
        if (expectedRingCount > 0) {
            logger.debug("*** Start of handling rings. ***");
            Cycles.markRingAtomsAndBonds(this.molecule);
            this.sssr = Cycles.sssr(this.molecule).toRingSet();
            if (this.sssr.getAtomContainerCount() < 1) {
                return;
            }
            AtomContainerSetManipulator.sort(this.sssr);
            this.markRingAtoms(this.sssr);
            this.ringPlacer.setMolecule(this.molecule);
            this.ringPlacer.checkAndMarkPlaced(this.sssr);
            this.ringSystems = RingPartitioner.partitionRings(this.sssr);
            Collections.sort(this.ringSystems, new Comparator<IRingSet>(){

                @Override
                public int compare(IRingSet a, IRingSet b) {
                    return -Integer.compare(AtomContainerSetManipulator.getBondCount(a), AtomContainerSetManipulator.getBondCount(b));
                }
            });
            int largest = 0;
            int numComplex = 0;
            int largestSize = this.ringSystems.get(0).getAtomContainerCount();
            if (largestSize > 1) {
                ++numComplex;
            }
            logger.debug("We have " + this.ringSystems.size() + " ring system(s).");
            for (int f = 1; f < this.ringSystems.size(); ++f) {
                logger.debug("RingSet " + f + " has size " + this.ringSystems.get(f).getAtomContainerCount());
                int size = this.ringSystems.get(f).getAtomContainerCount();
                if (size > 1) {
                    ++numComplex;
                }
                if (size <= largestSize) continue;
                largestSize = this.ringSystems.get(f).getAtomContainerCount();
                largest = f;
            }
            logger.debug("Largest RingSystem is at RingSet collection's position " + largest);
            logger.debug("Size of Largest RingSystem: " + largestSize);
            int respect = this.layoutRingSet(firstBondVector, this.ringSystems.get(largest));
            if (respect == 1 && numComplex == 1 || respect == 2) {
                this.selectOrientation = false;
            }
            logger.debug("First RingSet placed");
            this.ringPlacer.placeRingSubstituents(this.ringSystems.get(largest), this.bondLength);
        } else {
            logger.debug("*** Start of handling purely aliphatic molecules. ***");
            logger.debug("Searching initialLongestChain for this purely aliphatic molecule");
            IAtomContainer longestChain = AtomPlacer.getInitialLongestChain(this.molecule);
            logger.debug("Found linear chain of length " + longestChain.getAtomCount());
            logger.debug("Setting coordinated of first atom to 0,0");
            longestChain.getAtom(0).setPoint2d(new Point2d(0.0, 0.0));
            longestChain.getAtom(0).setFlag(1, true);
            double angle = Math.toRadians(-30.0);
            logger.debug("Attempting to place the first bond such that the whole chain will be horizontally alligned on the x axis");
            if (firstBondVector != null && firstBondVector != DEFAULT_BOND_VECTOR) {
                this.atomPlacer.placeLinearChain(longestChain, firstBondVector, this.bondLength);
            } else {
                this.atomPlacer.placeLinearChain(longestChain, new Vector2d(Math.cos(angle), Math.sin(angle)), this.bondLength);
            }
            logger.debug("Placed longest aliphatic chain");
        }
        do {
            logger.debug("*** Start of handling the rest of the molecule. ***");
            this.handleAliphatics();
            this.layoutNextRingSystem();
        } while (!AtomPlacer.allPlaced(this.molecule) && ++safetyCounter <= this.molecule.getAtomCount());
        if (!isSubLayout) {
            this.assignStereochem(this.molecule);
        }
        this.refinePlacement(this.molecule);
        this.finalizeLayout(this.molecule);
    }

    private void assignStereochem(IAtomContainer molecule) {
        CorrectGeometricConfiguration.correct(molecule);
        NonplanarBonds.assign(molecule);
    }

    private void refinePlacement(IAtomContainer molecule) {
        AtomPlacer.prioritise(molecule);
        LayoutRefiner refiner = new LayoutRefiner(molecule);
        refiner.refine();
        if (this.selectOrientation) {
            IAtom begAttach = null;
            for (IAtom atom : molecule.atoms()) {
                if (!(atom instanceof IPseudoAtom) || ((IPseudoAtom)atom).getAttachPointNum() != 1) continue;
                begAttach = atom;
                break;
            }
            if (begAttach == null) {
                StructureDiagramGenerator.selectOrientation(molecule, 3.0, 1);
            } else {
                List<IBond> attachBonds = molecule.getConnectedBondsList(begAttach);
                if (attachBonds.size() == 1) {
                    IAtom end = attachBonds.get(0).getConnectedAtom(begAttach);
                    Point2d xyBeg = begAttach.getPoint2d();
                    Point2d xyEnd = end.getPoint2d();
                    GeometryUtil.rotate(molecule, GeometryUtil.get2DCenter(molecule), -Math.atan2(xyEnd.y - xyBeg.y, xyEnd.x - xyBeg.x));
                    double ylo = 0.0;
                    double yhi = 0.0;
                    for (IAtom atom : molecule.atoms()) {
                        double yDelta = xyBeg.y - atom.getPoint2d().y;
                        if (yDelta > 0.0 && yDelta > yhi) {
                            yhi = yDelta;
                            continue;
                        }
                        if (!(yDelta < 0.0) || !(yDelta < ylo)) continue;
                        ylo = yDelta;
                    }
                    if (Math.abs(ylo) < yhi) {
                        for (IAtom atom : molecule.atoms()) {
                            atom.getPoint2d().y = -atom.getPoint2d().y;
                        }
                    }
                    GeometryUtil.rotate(molecule, GeometryUtil.get2DCenter(molecule), -Math.toRadians(30.0));
                }
            }
        }
    }

    private void finalizeLayout(IAtomContainer mol) {
        this.placeMultipleGroups(mol);
        this.placePositionalVariation(mol);
        this.placeSgroupBrackets(mol);
    }

    private static void selectOrientation(IAtomContainer mol, double widthDiff, int alignDiff) {
        int i;
        double[] minmax = GeometryUtil.getMinMax(mol);
        Point2d pivot = new Point2d(minmax[0] + (minmax[2] - minmax[0]) / 2.0, minmax[1] + (minmax[3] - minmax[1]) / 2.0);
        double maxWidth = minmax[2] - minmax[0];
        int maxAligned = StructureDiagramGenerator.countAlignedBonds(mol);
        Point2d[] coords = new Point2d[mol.getAtomCount()];
        for (int i2 = 0; i2 < mol.getAtomCount(); ++i2) {
            coords[i2] = new Point2d(mol.getAtom(i2).getPoint2d());
        }
        double step = Math.toRadians(30.0);
        int numSteps = 11;
        for (i = 0; i < 11; ++i) {
            int aligned;
            int alignDelta;
            GeometryUtil.rotate(mol, pivot, step);
            minmax = GeometryUtil.getMinMax(mol);
            double width = minmax[2] - minmax[0];
            double delta = Math.abs(width - maxWidth);
            if (delta > widthDiff && width > maxWidth) {
                maxWidth = width;
                for (int j = 0; j < mol.getAtomCount(); ++j) {
                    coords[j] = new Point2d(mol.getAtom(j).getPoint2d());
                }
                continue;
            }
            if (!(delta <= widthDiff) || (alignDelta = (aligned = StructureDiagramGenerator.countAlignedBonds(mol)) - maxAligned) <= alignDiff && (alignDelta != 0 || !(width > maxWidth))) continue;
            maxAligned = aligned;
            maxWidth = width;
            for (int j = 0; j < mol.getAtomCount(); ++j) {
                coords[j] = new Point2d(mol.getAtom(j).getPoint2d());
            }
        }
        for (i = 0; i < mol.getAtomCount(); ++i) {
            mol.getAtom(i).setPoint2d(coords[i]);
        }
    }

    private static int countAlignedBonds(IAtomContainer mol) {
        double ref = Math.toRadians(30.0);
        double diff = Math.toRadians(1.0);
        int count = 0;
        for (IBond bond : mol.bonds()) {
            Point2d beg = bond.getAtom(0).getPoint2d();
            Point2d end = bond.getAtom(1).getPoint2d();
            if (beg.x > end.x) {
                Point2d tmp = beg;
                beg = end;
                end = tmp;
            }
            Vector2d vec = new Vector2d(end.x - beg.x, end.y - beg.y);
            double angle = Math.atan2(vec.y, vec.x);
            if (!(Math.abs(angle) - ref < diff)) continue;
            ++count;
        }
        return count;
    }

    private void generateFragmentCoordinates(IAtomContainer mol, List<IAtomContainer> frags) throws CDKException {
        int i;
        List<IBond> ionicBonds = this.makeIonicBonds(frags);
        if (!ionicBonds.isEmpty()) {
            int rollback = mol.getBondCount();
            for (IBond bond : ionicBonds) {
                mol.addBond(bond);
            }
            frags = this.toList(ConnectivityChecker.partitionIntoMolecules(mol));
            int numBonds = mol.getBondCount();
            while (numBonds-- > rollback) {
                mol.removeBond(numBonds);
            }
        }
        ArrayList<double[]> limits = new ArrayList<double[]>();
        int numFragments = frags.size();
        for (IAtomContainer fragment : frags) {
            this.setMolecule(fragment, false);
            this.generateCoordinates(DEFAULT_BOND_VECTOR, true, true);
            limits.add(GeometryUtil.getMinMax(fragment));
        }
        int nRow = (int)Math.floor(Math.sqrt(numFragments));
        int nCol = (int)Math.ceil((double)numFragments / (double)nRow);
        double[] xOffsets = new double[nCol + 1];
        double[] yOffsets = new double[nRow + 1];
        double spacing = 1.5 * this.bondLength;
        for (i = 0; i < numFragments; ++i) {
            int col = 1 + i % nCol;
            int row = 1 + i / nCol;
            double[] minmax = (double[])limits.get(i);
            double width = spacing + (minmax[2] - minmax[0]);
            double height = spacing + (minmax[3] - minmax[1]);
            if (width > xOffsets[col]) {
                xOffsets[col] = width;
            }
            if (!(height > yOffsets[row])) continue;
            yOffsets[row] = height;
        }
        for (i = 1; i < xOffsets.length; ++i) {
            int n = i;
            xOffsets[n] = xOffsets[n] + xOffsets[i - 1];
        }
        for (i = 1; i < yOffsets.length; ++i) {
            int n = i;
            yOffsets[n] = yOffsets[n] + yOffsets[i - 1];
        }
        for (i = 0; i < limits.size(); ++i) {
            int row = nRow - i / nCol - 1;
            int col = i % nCol;
            Point2d dest = new Point2d((xOffsets[col] + xOffsets[col + 1]) / 2.0, (yOffsets[row] + yOffsets[row + 1]) / 2.0);
            double[] minmax = (double[])limits.get(i);
            Point2d curr = new Point2d((minmax[0] + minmax[2]) / 2.0, (minmax[1] + minmax[3]) / 2.0);
            GeometryUtil.translate2D(frags.get(i), dest.x - curr.x, dest.y - curr.y);
        }
        this.assignStereochem(mol);
        this.finalizeLayout(mol);
    }

    private List<IAtomContainer> mergeAtomicIons(List<IAtomContainer> frags) {
        ArrayList<IAtomContainer> res = new ArrayList<IAtomContainer>(frags.size());
        for (IAtomContainer frag : frags) {
            int i;
            IChemObjectBuilder bldr = frag.getBuilder();
            if (frag.getBondCount() > 0 || res.isEmpty()) {
                res.add(bldr.newInstance(IAtomContainer.class, frag));
                continue;
            }
            for (i = 0; i < res.size(); ++i) {
                IAtom iAtm = frag.getAtom(0);
                if (((IAtomContainer)res.get(i)).getBondCount() != 0) continue;
                IAtom jAtm = ((IAtomContainer)res.get(i)).getAtom(0);
                if (StructureDiagramGenerator.nullAsZero(iAtm.getFormalCharge()) == StructureDiagramGenerator.nullAsZero(jAtm.getFormalCharge()) && StructureDiagramGenerator.nullAsZero(iAtm.getAtomicNumber()) == StructureDiagramGenerator.nullAsZero(jAtm.getAtomicNumber()) && StructureDiagramGenerator.nullAsZero(iAtm.getImplicitHydrogenCount()) == StructureDiagramGenerator.nullAsZero(jAtm.getImplicitHydrogenCount())) break;
            }
            if (i < res.size()) {
                ((IAtomContainer)res.get(i)).add(frag);
                continue;
            }
            res.add(bldr.newInstance(IAtomContainer.class, frag));
        }
        return res;
    }

    private List<IAtom> selectIons(IAtomContainer frag, int sign) {
        int atmChg;
        int fragChg = (Integer)frag.getProperty(FRAGMENT_CHARGE);
        assert (Integer.signum(fragChg) == sign);
        ArrayList<IAtom> atoms = new ArrayList<IAtom>();
        block0: for (IAtom atom : frag.atoms()) {
            if (fragChg == 0) break;
            atmChg = StructureDiagramGenerator.nullAsZero(atom.getFormalCharge());
            if (Integer.signum(atmChg) != sign) continue;
            for (IBond bond : frag.getConnectedBondsList(atom)) {
                if (Integer.signum(StructureDiagramGenerator.nullAsZero(bond.getConnectedAtom(atom).getFormalCharge())) + sign != 0) continue;
                continue block0;
            }
            while (fragChg != 0 && atmChg != 0) {
                atoms.add(atom);
                atmChg -= sign;
                fragChg -= sign;
            }
        }
        if (fragChg == 0) {
            return atoms;
        }
        for (IAtom atom : frag.atoms()) {
            if (fragChg == 0) break;
            atmChg = StructureDiagramGenerator.nullAsZero(atom.getFormalCharge());
            if (Math.signum(atmChg) != (float)sign) continue;
            while (fragChg != 0 && atmChg != 0) {
                atoms.add(atom);
                atmChg -= sign;
                fragChg -= sign;
            }
        }
        return atoms;
    }

    private List<IBond> makeIonicBonds(List<IAtomContainer> frags) {
        assert (frags.size() > 1);
        HashSet remove = new HashSet();
        List<IAtomContainer> mergedFrags = this.mergeAtomicIons(frags);
        ArrayList<IAtomContainer> posFrags = new ArrayList<IAtomContainer>();
        ArrayList<IAtomContainer> negFrags = new ArrayList<IAtomContainer>();
        int chgSum = 0;
        for (IAtomContainer frag : mergedFrags) {
            int chg = 0;
            for (IAtom atom : frag.atoms()) {
                chg += StructureDiagramGenerator.nullAsZero(atom.getFormalCharge());
            }
            chgSum += chg;
            frag.setProperty(FRAGMENT_CHARGE, chg);
            if (chg < 0) {
                negFrags.add(frag);
                continue;
            }
            if (chg <= 0) continue;
            posFrags.add(frag);
        }
        if (chgSum != 0 || mergedFrags.size() == 1) {
            return Collections.emptyList();
        }
        ArrayList<IAtom> cations = new ArrayList<IAtom>();
        ArrayList<IAtom> anions = new ArrayList<IAtom>();
        HashMap atmMap = new HashMap();
        if (posFrags.size() == 1 && negFrags.size() == 1) {
            cations.addAll(this.selectIons((IAtomContainer)posFrags.get(0), 1));
            anions.addAll(this.selectIons((IAtomContainer)negFrags.get(0), -1));
        } else {
            Comparator<IAtomContainer> comparator = new Comparator<IAtomContainer>(){

                @Override
                public int compare(IAtomContainer a, IAtomContainer b) {
                    int qA = (Integer)a.getProperty(StructureDiagramGenerator.FRAGMENT_CHARGE);
                    int qB = (Integer)b.getProperty(StructureDiagramGenerator.FRAGMENT_CHARGE);
                    int cmp = Integer.compare(Math.abs(qA), Math.abs(qB));
                    if (cmp != 0) {
                        return cmp;
                    }
                    int sign = Integer.signum(qA);
                    return Integer.compare(sign * a.getBondCount(), sign * b.getBondCount());
                }
            };
            Collections.sort(posFrags, comparator);
            Collections.sort(negFrags, comparator);
            for (IAtomContainer posFrag : posFrags) {
                cations.addAll(this.selectIons(posFrag, 1));
            }
            for (IAtomContainer negFrag : negFrags) {
                anions.addAll(this.selectIons(negFrag, -1));
            }
        }
        if (cations.size() != anions.size() && cations.isEmpty()) {
            return Collections.emptyList();
        }
        IChemObjectBuilder bldr = frags.get(0).getBuilder();
        ArrayList<IBond> ionicBonds = new ArrayList<IBond>(cations.size());
        for (int i = 0; i < cations.size(); ++i) {
            IAtom beg = (IAtom)cations.get(i);
            IAtom end = (IAtom)anions.get(i);
            boolean unique = true;
            for (IBond bond : ionicBonds) {
                if ((!bond.getAtom(0).equals(beg) || !bond.getAtom(1).equals(end)) && (!bond.getAtom(1).equals(beg) || !bond.getAtom(0).equals(end))) continue;
                unique = false;
            }
            if (!unique) continue;
            ionicBonds.add(bldr.newInstance(IBond.class, beg, end));
        }
        return ionicBonds;
    }

    private static int nullAsZero(Integer x) {
        return x == null ? 0 : x;
    }

    private List<IAtomContainer> toList(IAtomContainerSet frags) {
        return new ArrayList<IAtomContainer>(FluentIterable.from(frags.atomContainers()).toList());
    }

    public void generateCoordinates() throws CDKException {
        this.generateCoordinates(DEFAULT_BOND_VECTOR);
    }

    private boolean lookupRingSystem(IRingSet rs, IAtomContainer molecule, boolean anon) {
        if (!this.useIdentTemplates) {
            return false;
        }
        IChemObjectBuilder bldr = molecule.getBuilder();
        IAtomContainer ringSystem = bldr.newInstance(IAtomContainer.class, new Object[0]);
        for (IAtomContainer container : rs.atomContainers()) {
            ringSystem.add(container);
        }
        HashSet<IAtom> ringAtoms = new HashSet<IAtom>();
        for (IAtom atom : ringSystem.atoms()) {
            ringAtoms.add(atom);
        }
        IAtomContainer ringWithStubs = bldr.newInstance(IAtomContainer.class, new Object[0]);
        ringWithStubs.add(ringSystem);
        for (IBond bond : molecule.bonds()) {
            IAtom atom1 = bond.getAtom(0);
            IAtom atom2 = bond.getAtom(1);
            if (StructureDiagramGenerator.isHydrogen(atom1) || StructureDiagramGenerator.isHydrogen(atom2) || !(ringAtoms.contains(atom1) ^ ringAtoms.contains(atom2))) continue;
            ringWithStubs.addBond(bond);
            ringWithStubs.addAtom(atom1);
            ringWithStubs.addAtom(atom2);
        }
        IAtomContainer skeletonStub = StructureDiagramGenerator.clearHydrogenCounts(AtomContainerManipulator.skeleton(ringWithStubs));
        IAtomContainer skeleton = StructureDiagramGenerator.clearHydrogenCounts(AtomContainerManipulator.skeleton(ringSystem));
        IAtomContainer anonymous = StructureDiagramGenerator.clearHydrogenCounts(AtomContainerManipulator.anonymise(ringSystem));
        for (IAtomContainer container : Arrays.asList(skeletonStub, skeleton, anonymous)) {
            if (!anon && container == anonymous || !this.identityLibrary.assignLayout(container)) continue;
            for (int i = 0; i < ringSystem.getAtomCount(); ++i) {
                IAtom atom = ringSystem.getAtom(i);
                atom.setPoint2d(container.getAtom(i).getPoint2d());
                atom.setFlag(1, true);
            }
            return true;
        }
        return false;
    }

    private static boolean isHydrogen(IAtom atom) {
        if (atom.getAtomicNumber() != null) {
            return atom.getAtomicNumber() == 1;
        }
        return "H".equals(atom.getSymbol());
    }

    private static IAtomContainer clearHydrogenCounts(IAtomContainer container) {
        for (IAtom atom : container.atoms()) {
            atom.setImplicitHydrogenCount(0);
        }
        return container;
    }

    private int layoutRingSet(Vector2d firstBondVector, IRingSet rs) {
        IAtomContainer ring2;
        RingSetManipulator.sort(rs);
        IRing first = RingSetManipulator.getMostComplexRing(rs);
        boolean macro = this.isMacroCycle(first, rs);
        int result = 0;
        if (this.lookupRingSystem(rs, this.molecule, !macro || rs.getAtomContainerCount() > 1)) {
            for (IAtomContainer container : rs.atomContainers()) {
                container.setFlag(1, true);
            }
            rs.setFlag(1, true);
            return macro ? 2 : 1;
        }
        IRingSet core = this.getRingSetCore(rs);
        if (core.getAtomContainerCount() > 0 && this.lookupRingSystem(core, this.molecule, !macro || rs.getAtomContainerCount() > 1)) {
            for (IAtomContainer container : core.atomContainers()) {
                container.setFlag(1, true);
            }
        }
        if (!first.getFlag(1)) {
            IAtomContainer sharedAtoms = this.placeFirstBond(first.getBond(0), firstBondVector);
            if (!macro || !this.macroPlacer.layout(first, rs)) {
                Vector2d ringCenterVector = this.ringPlacer.getRingCenterOfFirstRing(first, firstBondVector, this.bondLength);
                this.ringPlacer.placeRing(first, sharedAtoms, GeometryUtil.get2DCenter(sharedAtoms), ringCenterVector, this.bondLength);
            } else {
                result = 2;
            }
            first.setFlag(1, true);
        }
        if (macro) {
            for (IAtomContainer ring2 : rs.atomContainers()) {
                ring2.setProperty("sdg.snap.bridged", true);
            }
        }
        int thisRing = 0;
        ring2 = first;
        do {
            if (ring2.getFlag(1)) {
                this.ringPlacer.placeConnectedRings(rs, (IRing)ring2, RingPlacer.FUSED, this.bondLength);
                this.ringPlacer.placeConnectedRings(rs, (IRing)ring2, RingPlacer.BRIDGED, this.bondLength);
                this.ringPlacer.placeConnectedRings(rs, (IRing)ring2, RingPlacer.SPIRO, this.bondLength);
            }
            if (++thisRing == rs.getAtomContainerCount()) {
                thisRing = 0;
            }
            ring2 = (IRing)rs.getAtomContainer(thisRing);
        } while (!this.allPlaced(rs));
        return result;
    }

    private IRingSet getRingSetCore(IRingSet rs) {
        HashMultimap<IBond, IRing> ringlookup = HashMultimap.create();
        LinkedHashSet<IRing> ringsystem = new LinkedHashSet<IRing>();
        for (IAtomContainer ring : rs.atomContainers()) {
            ringsystem.add((IRing)ring);
            for (IBond bond : ring.bonds()) {
                ringlookup.put(bond, (IRing)ring);
            }
        }
        HashSet<IRing> toremove = new HashSet<IRing>();
        do {
            toremove.clear();
            for (IRing ring : ringsystem) {
                int numAttach = 0;
                for (IBond bond : ring.bonds()) {
                    for (IRing attached : ringlookup.get(bond)) {
                        if (attached == ring || !ringsystem.contains(attached)) continue;
                        ++numAttach;
                    }
                }
                if (numAttach > true) continue;
                toremove.add(ring);
            }
            ringsystem.removeAll(toremove);
        } while (!toremove.isEmpty());
        IRingSet core = rs.getBuilder().newInstance(IRingSet.class, new Object[0]);
        for (IRing ring : ringsystem) {
            core.addAtomContainer(ring);
        }
        return core;
    }

    private boolean isMacroCycle(IRing ring, IRingSet rs) {
        if (ring.getAtomCount() < 10) {
            return false;
        }
        for (IBond bond : ring.bonds()) {
            boolean found = false;
            for (IAtomContainer other : rs.atomContainers()) {
                if (ring == other || !other.contains(bond)) continue;
                found = true;
                break;
            }
            if (found) continue;
            return true;
        }
        return false;
    }

    private void handleAliphatics() throws CDKException {
        boolean done;
        logger.debug("Start of handleAliphatics");
        int safetyCounter = 0;
        IAtomContainer unplacedAtoms = null;
        IAtomContainer placedAtoms = null;
        IAtomContainer longestUnplacedChain = null;
        IAtom atom = null;
        Vector2d direction = null;
        Vector2d startVector = null;
        do {
            ++safetyCounter;
            done = false;
            atom = this.getNextAtomWithAliphaticUnplacedNeigbors();
            if (atom != null) {
                unplacedAtoms = this.getUnplacedAtoms(atom);
                placedAtoms = this.getPlacedAtoms(atom);
                longestUnplacedChain = AtomPlacer.getLongestUnplacedChain(this.molecule, atom);
                logger.debug("---start of longest unplaced chain---");
                try {
                    logger.debug("Start at atom no. " + (this.molecule.getAtomNumber(atom) + 1));
                    logger.debug(AtomPlacer.listNumbers(this.molecule, longestUnplacedChain));
                }
                catch (Exception exc) {
                    logger.debug(exc);
                }
                logger.debug("---end of longest unplaced chain---");
                if (longestUnplacedChain.getAtomCount() > 1) {
                    if (placedAtoms.getAtomCount() > 1) {
                        logger.debug("More than one atoms placed already");
                        logger.debug("trying to place neighbors of atom " + (this.molecule.getAtomNumber(atom) + 1));
                        this.atomPlacer.distributePartners(atom, placedAtoms, GeometryUtil.get2DCenter(placedAtoms), unplacedAtoms, this.bondLength);
                        direction = new Vector2d(longestUnplacedChain.getAtom(1).getPoint2d());
                        startVector = new Vector2d(atom.getPoint2d());
                        direction.sub(startVector);
                        logger.debug("Done placing neighbors of atom " + (this.molecule.getAtomNumber(atom) + 1));
                    } else {
                        logger.debug("Less than or equal one atoms placed already");
                        logger.debug("Trying to get next bond vector.");
                        direction = this.atomPlacer.getNextBondVector(atom, placedAtoms.getAtom(0), GeometryUtil.get2DCenter(this.molecule), true);
                    }
                    for (int f = 1; f < longestUnplacedChain.getAtomCount(); ++f) {
                        longestUnplacedChain.getAtom(f).setFlag(1, false);
                    }
                    this.atomPlacer.placeLinearChain(longestUnplacedChain, direction, this.bondLength);
                    continue;
                }
                done = true;
                continue;
            }
            done = true;
        } while (!done && safetyCounter <= this.molecule.getAtomCount());
        logger.debug("End of handleAliphatics");
    }

    private void layoutNextRingSystem() throws CDKException {
        logger.debug("Start of layoutNextRingSystem()");
        this.resetUnplacedRings();
        IAtomContainer tempAc = AtomPlacer.getPlacedAtoms(this.molecule);
        logger.debug("Finding attachment bond to already placed part...");
        IBond nextRingAttachmentBond = this.getNextBondWithUnplacedRingAtom();
        if (nextRingAttachmentBond != null) {
            logger.debug("...bond found.");
            IAtom ringAttachmentAtom = this.getRingAtom(nextRingAttachmentBond);
            IAtom chainAttachmentAtom = this.getOtherBondAtom(ringAttachmentAtom, nextRingAttachmentBond);
            IRingSet nextRingSystem = this.getRingSystemOfAtom(this.ringSystems, ringAttachmentAtom);
            IAtomContainer ringSystem = tempAc.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
            Iterator<IAtomContainer> containers = RingSetManipulator.getAllAtomContainers(nextRingSystem).iterator();
            while (containers.hasNext()) {
                ringSystem.add(containers.next());
            }
            Point2d oldRingAttachmentAtomPoint = ringAttachmentAtom.getPoint2d();
            Point2d oldChainAttachmentAtomPoint = chainAttachmentAtom.getPoint2d();
            this.layoutRingSet(this.firstBondVector, nextRingSystem);
            AtomPlacer.markNotPlaced(tempAc);
            IAtomContainer placedRingSubstituents = this.ringPlacer.placeRingSubstituents(nextRingSystem, this.bondLength);
            ringSystem.add(placedRingSubstituents);
            AtomPlacer.markPlaced(tempAc);
            logger.debug("Computing translation/rotation of new ringset to fit old attachment bond orientation...");
            Point2d oldPoint2 = oldRingAttachmentAtomPoint;
            Point2d oldPoint1 = oldChainAttachmentAtomPoint;
            Point2d newPoint2 = ringAttachmentAtom.getPoint2d();
            Point2d newPoint1 = chainAttachmentAtom.getPoint2d();
            logger.debug("oldPoint1: " + oldPoint1);
            logger.debug("oldPoint2: " + oldPoint2);
            logger.debug("newPoint1: " + newPoint1);
            logger.debug("newPoint2: " + newPoint2);
            double oldAngle = GeometryUtil.getAngle(oldPoint2.x - oldPoint1.x, oldPoint2.y - oldPoint1.y);
            double newAngle = GeometryUtil.getAngle(newPoint2.x - newPoint1.x, newPoint2.y - newPoint1.y);
            double angleDiff = oldAngle - newAngle;
            logger.debug("oldAngle: " + oldAngle + ", newAngle: " + newAngle + "; diff = " + angleDiff);
            Vector2d translationVector = new Vector2d(oldPoint1);
            translationVector.sub(new Vector2d(newPoint1));
            GeometryUtil.translate2D(ringSystem, translationVector);
            GeometryUtil.rotate(ringSystem, oldPoint1, angleDiff);
            logger.debug("...done translating/rotating new ringset to fit old attachment bond orientation.");
        } else {
            logger.debug("...no bond found");
        }
        logger.debug("End of layoutNextRingSystem()");
    }

    private IAtomContainer getUnplacedAtoms(IAtom atom) {
        IAtomContainer unplacedAtoms = atom.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
        List<IBond> bonds = this.molecule.getConnectedBondsList(atom);
        for (int f = 0; f < bonds.size(); ++f) {
            IAtom connectedAtom = bonds.get(f).getConnectedAtom(atom);
            if (connectedAtom.getFlag(1)) continue;
            unplacedAtoms.addAtom(connectedAtom);
        }
        return unplacedAtoms;
    }

    private IAtomContainer getPlacedAtoms(IAtom atom) {
        IAtomContainer placedAtoms = atom.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
        List<IBond> bonds = this.molecule.getConnectedBondsList(atom);
        for (int f = 0; f < bonds.size(); ++f) {
            IAtom connectedAtom = bonds.get(f).getConnectedAtom(atom);
            if (!connectedAtom.getFlag(1)) continue;
            placedAtoms.addAtom(connectedAtom);
        }
        return placedAtoms;
    }

    private IAtom getNextAtomWithAliphaticUnplacedNeigbors() {
        for (int f = 0; f < this.molecule.getBondCount(); ++f) {
            IBond bond = this.molecule.getBond(f);
            if (bond.getAtom(1).getFlag(1) && !bond.getAtom(0).getFlag(1)) {
                return bond.getAtom(1);
            }
            if (!bond.getAtom(0).getFlag(1) || bond.getAtom(1).getFlag(1)) continue;
            return bond.getAtom(0);
        }
        return null;
    }

    private IBond getNextBondWithUnplacedRingAtom() {
        for (IBond bond : this.molecule.bonds()) {
            if (bond.getAtom(0).getPoint2d() == null || bond.getAtom(1).getPoint2d() == null) continue;
            if (bond.getAtom(1).getFlag(1) && !bond.getAtom(0).getFlag(1) && bond.getAtom(0).getFlag(2)) {
                return bond;
            }
            if (!bond.getAtom(0).getFlag(1) || bond.getAtom(1).getFlag(1) || !bond.getAtom(1).getFlag(2)) continue;
            return bond;
        }
        return null;
    }

    private IAtomContainer placeFirstBond(IBond bond, Vector2d bondVector) {
        IAtomContainer sharedAtoms = null;
        try {
            bondVector.normalize();
            logger.debug("placeFirstBondOfFirstRing->bondVector.length():" + bondVector.length());
            bondVector.scale(this.bondLength);
            logger.debug("placeFirstBondOfFirstRing->bondVector.length() after scaling:" + bondVector.length());
            Point2d point = new Point2d(0.0, 0.0);
            IAtom atom = bond.getAtom(0);
            logger.debug("Atom 1 of first Bond: " + (this.molecule.getAtomNumber(atom) + 1));
            atom.setPoint2d(point);
            atom.setFlag(1, true);
            point = new Point2d(0.0, 0.0);
            atom = bond.getAtom(1);
            logger.debug("Atom 2 of first Bond: " + (this.molecule.getAtomNumber(atom) + 1));
            point.add(bondVector);
            atom.setPoint2d(point);
            atom.setFlag(1, true);
            sharedAtoms = atom.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
            sharedAtoms.addBond(bond);
            sharedAtoms.addAtom(bond.getAtom(0));
            sharedAtoms.addAtom(bond.getAtom(1));
        }
        catch (Exception exc) {
            logger.debug(exc);
        }
        return sharedAtoms;
    }

    private boolean allPlaced(IRingSet rings) {
        for (int f = 0; f < rings.getAtomContainerCount(); ++f) {
            if (((IRing)rings.getAtomContainer(f)).getFlag(1)) continue;
            logger.debug("allPlaced->Ring " + f + " not placed");
            return false;
        }
        return true;
    }

    private void markRingAtoms(IRingSet rings) {
        IRing ring = null;
        for (int i = 0; i < rings.getAtomContainerCount(); ++i) {
            ring = (IRing)rings.getAtomContainer(i);
            for (int j = 0; j < ring.getAtomCount(); ++j) {
                ring.getAtom(j).setFlag(2, true);
            }
        }
    }

    private IAtom getRingAtom(IBond bond) {
        if (bond.getAtom(0).getFlag(2) && !bond.getAtom(0).getFlag(1)) {
            return bond.getAtom(0);
        }
        if (bond.getAtom(1).getFlag(2) && !bond.getAtom(1).getFlag(1)) {
            return bond.getAtom(1);
        }
        return null;
    }

    private IRingSet getRingSystemOfAtom(List ringSystems, IAtom ringAtom) {
        IRingSet ringSet = null;
        for (int f = 0; f < ringSystems.size(); ++f) {
            ringSet = (IRingSet)ringSystems.get(f);
            if (!ringSet.contains(ringAtom)) continue;
            return ringSet;
        }
        return null;
    }

    private void resetUnplacedRings() {
        IRing ring = null;
        if (this.sssr == null) {
            return;
        }
        int unplacedCounter = 0;
        for (int f = 0; f < this.sssr.getAtomContainerCount(); ++f) {
            ring = (IRing)this.sssr.getAtomContainer(f);
            if (ring.getFlag(1)) continue;
            logger.debug("Ring with " + ring.getAtomCount() + " atoms is not placed.");
            ++unplacedCounter;
            for (int g = 0; g < ring.getAtomCount(); ++g) {
                ring.getAtom(g).setFlag(1, false);
            }
        }
        logger.debug("There are " + unplacedCounter + " unplaced Rings.");
    }

    public void setBondLength(double bondLength) {
        this.bondLength = bondLength;
    }

    public double getBondLength() {
        return this.bondLength;
    }

    public IAtom getOtherBondAtom(IAtom atom, IBond bond) {
        if (!bond.contains(atom)) {
            return null;
        }
        if (bond.getAtom(0).equals(atom)) {
            return bond.getAtom(1);
        }
        return bond.getAtom(0);
    }

    private void placeMultipleGroups(IAtomContainer mol) {
        List sgroups = (List)mol.getProperty("cdk:CtabSgroups");
        if (sgroups == null) {
            return;
        }
        ArrayList<Sgroup> multipleGroups = new ArrayList<Sgroup>();
        for (Sgroup sgroup : sgroups) {
            if (sgroup.getType() != SgroupType.CtabMultipleGroup) continue;
            multipleGroups.add(sgroup);
        }
        if (multipleGroups.isEmpty()) {
            return;
        }
        int[][] adjlist = GraphUtil.toAdjList(mol);
        HashMap<IAtom, Integer> idxs = new HashMap<IAtom, Integer>();
        for (IAtom atom : mol.atoms()) {
            idxs.put(atom, idxs.size());
        }
        for (Sgroup sgroup : multipleGroups) {
            int numCrossing = sgroup.getBonds().size();
            if (numCrossing != 0 && numCrossing != 2) continue;
            IAtomContainer substructure = mol.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
            final HashSet<IAtom> visit = new HashSet<IAtom>();
            Collection patoms = (Collection)sgroup.getValue(SgroupKey.CtabParentAtomList);
            if (patoms == null) continue;
            for (IAtom atom : patoms) {
                substructure.addAtom(atom);
                visit.add(atom);
            }
            for (IBond bond : mol.bonds()) {
                IAtom beg = bond.getAtom(0);
                IAtom end = bond.getAtom(1);
                if (!visit.contains(beg) || !visit.contains(end)) continue;
                substructure.addBond(bond);
            }
            visit.addAll(sgroup.getAtoms());
            Pattern ptrn = VentoFoggia.findSubstructure(substructure, new AtomMatcher(){

                @Override
                public boolean matches(IAtom a, IAtom b) {
                    int bHcnt;
                    int bMass;
                    int bChg;
                    int bElem;
                    if (!visit.contains(b)) {
                        return false;
                    }
                    int aElem = StructureDiagramGenerator.this.safeUnbox(a.getAtomicNumber());
                    if (aElem != (bElem = StructureDiagramGenerator.this.safeUnbox(b.getAtomicNumber()))) {
                        return false;
                    }
                    int aChg = StructureDiagramGenerator.this.safeUnbox(a.getFormalCharge());
                    if (aChg != (bChg = StructureDiagramGenerator.this.safeUnbox(b.getFormalCharge()))) {
                        return false;
                    }
                    int aMass = StructureDiagramGenerator.this.safeUnbox(a.getMassNumber());
                    if (aMass != (bMass = StructureDiagramGenerator.this.safeUnbox(b.getMassNumber()))) {
                        return false;
                    }
                    int aHcnt = StructureDiagramGenerator.this.safeUnbox(a.getImplicitHydrogenCount());
                    return aHcnt == (bHcnt = StructureDiagramGenerator.this.safeUnbox(b.getImplicitHydrogenCount()));
                }
            }, BondMatcher.forOrder());
            Set<IAtom> sgroupAtoms = sgroup.getAtoms();
            ArrayList<AbstractMap.SimpleImmutableEntry<Point2d, Vector2d>> outgoing = new ArrayList<AbstractMap.SimpleImmutableEntry<Point2d, Vector2d>>();
            ArrayList<AbstractMap.SimpleImmutableEntry<IBond, Vector2d>> xBondVec = new ArrayList<AbstractMap.SimpleImmutableEntry<IBond, Vector2d>>();
            if (numCrossing == 2) {
                IAtom end;
                IAtom beg;
                for (IBond iBond : mol.bonds()) {
                    beg = iBond.getAtom(0);
                    end = iBond.getAtom(1);
                    if (patoms.contains(beg) == patoms.contains(end)) continue;
                    if (patoms.contains(beg)) {
                        outgoing.add(new AbstractMap.SimpleImmutableEntry<Point2d, Vector2d>(beg.getPoint2d(), new Vector2d(end.getPoint2d().x - beg.getPoint2d().x, end.getPoint2d().y - beg.getPoint2d().y)));
                        continue;
                    }
                    outgoing.add(new AbstractMap.SimpleImmutableEntry<Point2d, Vector2d>(end.getPoint2d(), new Vector2d(beg.getPoint2d().x - end.getPoint2d().x, beg.getPoint2d().y - end.getPoint2d().y)));
                }
                for (IBond iBond : sgroup.getBonds()) {
                    beg = iBond.getAtom(0);
                    end = iBond.getAtom(1);
                    if (sgroupAtoms.contains(beg)) {
                        xBondVec.add(new AbstractMap.SimpleImmutableEntry<IBond, Vector2d>(iBond, new Vector2d(end.getPoint2d().x - beg.getPoint2d().x, end.getPoint2d().y - beg.getPoint2d().y)));
                        continue;
                    }
                    xBondVec.add(new AbstractMap.SimpleImmutableEntry<IBond, Vector2d>(iBond, new Vector2d(beg.getPoint2d().x - end.getPoint2d().x, beg.getPoint2d().y - end.getPoint2d().y)));
                }
            }
            visit.removeAll(patoms);
            for (Map map : ptrn.matchAll(mol).uniqueAtoms().toAtomMap()) {
                for (Map.Entry e : map.entrySet()) {
                    ((IAtom)e.getValue()).setPoint2d(new Point2d(((IAtom)e.getKey()).getPoint2d()));
                }
                visit.removeAll(map.values());
            }
            assert (xBondVec.size() == outgoing.size());
            for (Map.Entry entry : xBondVec) {
                IBond bond = (IBond)entry.getKey();
                if (bond.isInRing()) continue;
                IAtom beg = sgroupAtoms.contains(bond.getAtom(0)) ? bond.getAtom(0) : bond.getAtom(1);
                Map.Entry best = null;
                for (Map.Entry entry2 : outgoing) {
                    if (best != null && !(((Point2d)entry2.getKey()).distance(beg.getPoint2d()) < ((Point2d)best.getKey()).distance(beg.getPoint2d()))) continue;
                    best = entry2;
                }
                outgoing.remove(best);
                assert (best != null);
                HashSet<Integer> iVisit = new HashSet<Integer>();
                iVisit.add((Integer)idxs.get(beg));
                StructureDiagramGenerator.visit(iVisit, adjlist, (Integer)idxs.get(bond.getConnectedAtom(beg)));
                iVisit.remove(idxs.get(beg));
                IAtomContainer iAtomContainer = mol.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
                for (Integer idx : iVisit) {
                    iAtomContainer.addAtom(mol.getAtom(idx));
                }
                Vector2d orgVec = (Vector2d)entry.getValue();
                Vector2d newVec = (Vector2d)best.getValue();
                Point2d endP = bond.getConnectedAtom(beg).getPoint2d();
                Point2d newEndP = new Point2d(beg.getPoint2d());
                newEndP.add(newVec);
                double pDot = orgVec.x * newVec.y - orgVec.y * newVec.x;
                double theta = Math.atan2(pDot, newVec.dot(orgVec));
                GeometryUtil.translate2D(iAtomContainer, newEndP.x - endP.x, newEndP.y - endP.y);
                GeometryUtil.rotate(iAtomContainer, new Point2d(bond.getConnectedAtom(beg).getPoint2d()), theta);
            }
        }
    }

    private int safeUnbox(Integer x) {
        return x == null ? 0 : x;
    }

    private void placePositionalVariation(IAtomContainer mol) {
        List sgroups = (List)mol.getProperty("cdk:CtabSgroups");
        if (sgroups == null) {
            return;
        }
        Multimap<Set<IAtom>, IAtom> mapping = StructureDiagramGenerator.aggregateMulticenterSgroups(sgroups);
        if (mapping.isEmpty()) {
            return;
        }
        GraphUtil.EdgeToBondMap bondMap = GraphUtil.EdgeToBondMap.withSpaceFor(mol);
        int[][] adjlist = GraphUtil.toAdjList(mol, bondMap);
        HashMap<IAtom, Integer> idxs = new HashMap<IAtom, Integer>();
        for (IAtom iAtom : mol.atoms()) {
            idxs.put(iAtom, idxs.size());
        }
        for (Map.Entry entry : mapping.asMap().entrySet()) {
            LinkedHashSet<IBond> bonds = new LinkedHashSet<IBond>();
            IAtomContainer shared = mol.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
            for (IAtom atom : (Set)entry.getKey()) {
                shared.addAtom(atom);
            }
            Point2d center = GeometryUtil.get2DCenter(shared);
            for (IBond bond : mol.bonds()) {
                if (!((Set)entry.getKey()).contains(bond.getAtom(0)) || !((Set)entry.getKey()).contains(bond.getAtom(1))) continue;
                bonds.add(bond);
            }
            if (bonds.size() >= ((Collection)entry.getValue()).size()) {
                Iterator begIter = ((Collection)entry.getValue()).iterator();
                Iterator bndIter = bonds.iterator();
                while (begIter.hasNext() && bndIter.hasNext()) {
                    IBond bond = (IBond)bndIter.next();
                    IAtom atom = (IAtom)begIter.next();
                    if (StructureDiagramGenerator.numRingBonds(mol, bond.getAtom(0)) > 2 && StructureDiagramGenerator.numRingBonds(mol, bond.getAtom(1)) > 2) continue;
                    Point2d newBegP = new Point2d(bond.getAtom(0).getPoint2d());
                    Point2d newEndP = new Point2d(bond.getAtom(1).getPoint2d());
                    Vector2d bndVec = new Vector2d(newEndP.x - newBegP.x, newEndP.y - newBegP.y);
                    Vector2d bndXVec = new Vector2d(-bndVec.y, bndVec.x);
                    Vector2d centerVec = new Vector2d(center.x - (newBegP.x + newEndP.x) / 2.0, center.y - (newBegP.y + newEndP.y) / 2.0);
                    if (bndXVec.dot(centerVec) > 0.0) {
                        bndXVec.negate();
                    }
                    bndVec.normalize();
                    bndXVec.normalize();
                    bndVec.scale(0.5 * this.bondLength);
                    double bndStep = this.bondLength / 5.0;
                    newBegP.add(bndVec);
                    bndXVec.normalize();
                    bndXVec.scale(2.0 * bndStep);
                    newBegP.sub(bndXVec);
                    newEndP.sub(bndVec);
                    bndXVec.normalize();
                    bndXVec.scale(3.0 * bndStep);
                    newEndP.add(bndXVec);
                    int atomIdx = (Integer)idxs.get(atom);
                    if (adjlist[atomIdx].length != 1) continue;
                    HashSet<Integer> visited = new HashSet<Integer>();
                    StructureDiagramGenerator.visit(visited, adjlist, atomIdx);
                    IAtomContainer frag = mol.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
                    for (Integer visit : visited) {
                        frag.addAtom(mol.getAtom(visit));
                    }
                    IBond attachBond = bondMap.get(atomIdx, adjlist[atomIdx][0]);
                    Point2d begP = atom.getPoint2d();
                    Point2d endP = attachBond.getConnectedAtom(atom).getPoint2d();
                    Vector2d orgVec = new Vector2d(endP.x - begP.x, endP.y - begP.y);
                    Vector2d newVec = new Vector2d(newEndP.x - newBegP.x, newEndP.y - newBegP.y);
                    double pDot = orgVec.x * newVec.y - orgVec.y * newVec.x;
                    double theta = Math.atan2(pDot, newVec.dot(orgVec));
                    GeometryUtil.translate2D(frag, newBegP.x - begP.x, newBegP.y - begP.y);
                    GeometryUtil.rotate(frag, new Point2d(atom.getPoint2d()), theta);
                    frag.removeAtom(atom);
                    GeometryUtil.translate2D(frag, newEndP.x - endP.x, newEndP.y - endP.y);
                }
                continue;
            }
            System.err.println("Positional variation not yet handled");
        }
    }

    private static void visit(Set<Integer> visited, int[][] g, int v) {
        visited.add(v);
        for (int w : g[v]) {
            if (visited.contains(w)) continue;
            StructureDiagramGenerator.visit(visited, g, w);
        }
    }

    private static Multimap<Set<IAtom>, IAtom> aggregateMulticenterSgroups(List<Sgroup> sgroups) {
        HashMultimap<Set<IAtom>, IAtom> mapping = HashMultimap.create();
        for (Sgroup sgroup : sgroups) {
            if (sgroup.getType() != SgroupType.ExtMulticenter) continue;
            IAtom beg = null;
            HashSet<IAtom> ends = new HashSet<IAtom>();
            Set<IBond> bonds = sgroup.getBonds();
            if (bonds.size() != 1) continue;
            IBond bond = bonds.iterator().next();
            for (IAtom atom : sgroup.getAtoms()) {
                if (bond.contains(atom)) {
                    beg = atom;
                    continue;
                }
                ends.add(atom);
            }
            if (beg == null || ends.isEmpty()) continue;
            mapping.put(ends, beg);
        }
        return mapping;
    }

    private static int numRingBonds(IAtomContainer mol, IAtom atom) {
        int cnt = 0;
        for (IBond bond : mol.getConnectedBondsList(atom)) {
            if (!bond.isInRing()) continue;
            ++cnt;
        }
        return cnt;
    }

    private void placeSgroupBrackets(IAtomContainer mol) {
        ArrayList sgroups = (ArrayList)mol.getProperty("cdk:CtabSgroups");
        if (sgroups == null) {
            return;
        }
        HashMultimap<IBond, Sgroup> bondMap = HashMultimap.create();
        HashMap<IBond, Integer> counter = new HashMap<IBond, Integer>();
        for (Sgroup sgroup : sgroups) {
            if (!StructureDiagramGenerator.hasBrackets(sgroup)) continue;
            for (IBond bond : sgroup.getBonds()) {
                bondMap.put(bond, sgroup);
                counter.put(bond, 0);
            }
        }
        sgroups = new ArrayList(sgroups);
        Collections.sort(sgroups, new Comparator<Sgroup>(){

            @Override
            public int compare(Sgroup o1, Sgroup o2) {
                if (o1.getParents().isEmpty() != o2.getParents().isEmpty()) {
                    if (o1.getParents().isEmpty()) {
                        return 1;
                    }
                    return -1;
                }
                return 0;
            }
        });
        for (Sgroup sgroup : sgroups) {
            if (!StructureDiagramGenerator.hasBrackets(sgroup)) continue;
            Set<IAtom> atoms = sgroup.getAtoms();
            Set<IBond> xbonds = sgroup.getBonds();
            sgroup.putValue(SgroupKey.CtabBracket, null);
            if (xbonds.size() >= 2) {
                boolean vert = true;
                for (IBond bond : xbonds) {
                    double theta = StructureDiagramGenerator.angle(bond);
                    if (!(Math.abs(Math.toDegrees(theta)) > 40.0) || !(Math.abs(Math.toDegrees(theta)) < 140.0)) continue;
                    vert = false;
                    break;
                }
                for (IBond bond : xbonds) {
                    sgroup.addBracket(this.newCrossingBracket(bond, bondMap, counter, vert));
                }
                continue;
            }
            IAtomContainer tmp = mol.getBuilder().newInstance(IAtomContainer.class, new Object[0]);
            for (IAtom atom : atoms) {
                tmp.addAtom(atom);
            }
            double[] minmax = GeometryUtil.getMinMax(tmp);
            double padding = 0.7 * this.bondLength;
            sgroup.addBracket(new SgroupBracket(minmax[0] - padding, minmax[1] - padding, minmax[0] - padding, minmax[3] + padding));
            sgroup.addBracket(new SgroupBracket(minmax[2] + padding, minmax[1] - padding, minmax[2] + padding, minmax[3] + padding));
        }
    }

    private static double angle(IBond bond) {
        Point2d end = bond.getAtom(0).getPoint2d();
        Point2d beg = bond.getAtom(1).getPoint2d();
        return Math.atan2(end.y - beg.y, end.x - beg.x);
    }

    private SgroupBracket newCrossingBracket(IBond bond, Multimap<IBond, Sgroup> bonds, Map<IBond, Integer> counter, boolean vert) {
        IAtom beg = bond.getAtom(0);
        IAtom end = bond.getAtom(1);
        Point2d begXy = beg.getPoint2d();
        Point2d endXy = end.getPoint2d();
        Vector2d lenOffset = new Vector2d(endXy.x - begXy.x, endXy.y - begXy.y);
        Vector2d bndCrossVec = new Vector2d(-lenOffset.y, lenOffset.x);
        lenOffset.normalize();
        bndCrossVec.normalize();
        bndCrossVec.scale(0.9 * this.bondLength / 2.0);
        ArrayList<Sgroup> sgroups = new ArrayList<Sgroup>(bonds.get(bond));
        if (sgroups.size() == 1) {
            lenOffset.scale(0.5 * this.bondLength);
        } else if (sgroups.size() == 2) {
            boolean flip;
            boolean bl = flip = !((Sgroup)sgroups.get(counter.get(bond))).getAtoms().contains(beg);
            if (counter.get(bond) == 0) {
                lenOffset.scale(flip ? 0.75 : 0.25 * this.bondLength);
                counter.put(bond, 1);
            } else {
                lenOffset.scale(flip ? 0.25 : 0.75 * this.bondLength);
            }
        } else {
            double step = this.bondLength / (double)(1 + sgroups.size());
            int idx = counter.get(bond) + 1;
            counter.put(bond, idx);
            lenOffset.scale((double)idx * step * this.bondLength);
        }
        if (vert) {
            return new SgroupBracket(begXy.x + lenOffset.x, begXy.y + lenOffset.y + bndCrossVec.length(), begXy.x + lenOffset.x, begXy.y + lenOffset.y - bndCrossVec.length());
        }
        return new SgroupBracket(begXy.x + lenOffset.x + bndCrossVec.x, begXy.y + lenOffset.y + bndCrossVec.y, begXy.x + lenOffset.x - bndCrossVec.x, begXy.y + lenOffset.y - bndCrossVec.y);
    }

    private static boolean hasBrackets(Sgroup sgroup) {
        switch (sgroup.getType()) {
            case CtabStructureRepeatUnit: 
            case CtabAnyPolymer: 
            case CtabCrossLink: 
            case CtabComponent: 
            case CtabMixture: 
            case CtabFormulation: 
            case CtabGraft: 
            case CtabModified: 
            case CtabMonomer: 
            case CtabCopolymer: 
            case CtabMultipleGroup: {
                return true;
            }
            case CtabGeneric: {
                List brackets = (List)sgroup.getValue(SgroupKey.CtabBracket);
                return brackets != null && !brackets.isEmpty();
            }
        }
        return false;
    }
}

